<?php

namespace app\Http\Controllers;

use App\ControlAccount;
use App\MeasurementUnit;
use App\ServiceVoucher;
use App\Suppliers;
use Illuminate\Http\Request;

use Illuminate\Support\Facades\DB;

class ServiceVoucherController extends Controller
{

    protected string $module = 'Service Vouchers';

    public function __construct()
    {
        $module = $this->module;

        $this->middleware("permission:$module list")->only(['index']);
        $this->middleware("permission:$module view")->only(['show']);
        $this->middleware("permission:$module add")->only(['create', 'store']);
        $this->middleware("permission:$module edit")->only(['edit', 'update']);
        $this->middleware("permission:$module delete")->only(['destroy']);
    }
    public function index()
    {
        $vouchers  = ServiceVoucher::with('details', 'supplier', 'details.controlAccount')->get();
        // dd($vouchers);
        return view('service_vouchers.index', compact('vouchers'));
    }
    public function create()
    {
        // dd('sd');
        $suppliers = Suppliers::where('status', 1)->get();
        $accounts = ControlAccount::where('main_account_id', 4)->get();
        $units = MeasurementUnit::where('measurement_unit_status', '1')->get();
        // dd($units);
        return view('service_vouchers.create', compact('suppliers', 'accounts', 'units'));
    }

    public function store(Request $request)
    {
        // dd($request);
        $validated = $request->validate([
            'supplier_id'   => 'required|integer',
            'voucher_date'  => 'required|date',

            'truck_no'      => 'required|string|max:50',
            'bill_no'       => 'required|string|max:50',
            'description'   => 'nullable|string',

            'details'       => 'required|array|min:1',
            'details.*.control_account_id' => 'required|integer',
            'details.*.qty'                => 'required|numeric|min:0',
            'details.*.unit'               => 'required|min:1',
            'details.*.price'              => 'required|numeric|min:0',
            'details.*.narration'          => 'nullable|string',
        ]);

        DB::transaction(function () use ($validated) {

            $lastId = ServiceVoucher::max('id') + 1;
            $voucherNo = 'SV-' . date('Y') . '-' . str_pad($lastId, 5, '0', STR_PAD_LEFT);
            $totalAmount = collect($validated['details'])->sum(function ($item) {
                return $item['qty'] * $item['price'];
            });
            // dd($totalAmount);
            $voucher = ServiceVoucher::create([
                'voucher_no'   => $voucherNo,
                'voucher_date' => $validated['voucher_date'],
                'truck_no'     => $validated['truck_no'],
                'bill_no'      => $validated['bill_no'],
                'supplier_id'  => $validated['supplier_id'],
                'total_amount'  => $totalAmount,
                'description'  => $validated['description'] ?? null,
            ]);

            foreach ($validated['details'] as $row) {
                $voucher->details()->create([
                    'control_account_id' => $row['control_account_id'],
                    'qty'        => $row['qty'],
                    'unit'       => $row['unit'],
                    'price'      => $row['price'],
                    'amount'     => $row['qty'] * $row['price'],
                    'narration'  => $row['narration'] ?? null,
                ]);
            }
        });

        return redirect(route('service_vouchers.index'))->with('success', 'Service Voucher Created Successfully');
    }
    public function edit($id)
    {
        $id = decrypt($id);
        $voucher = ServiceVoucher::with('details')->findOrFail($id);
        $suppliers = Suppliers::all();
        $accounts  = ControlAccount::all();
        $units     = MeasurementUnit::all();

        return view('service_vouchers.edit', compact('voucher', 'suppliers', 'accounts', 'units'));
    }

    public function update(Request $request, $id)
    {
        $id = decrypt($id);
        $voucher = ServiceVoucher::findOrFail($id);

        $validated = $request->validate([
            'supplier_id'                  => 'required|integer',
            'voucher_date'                 => 'required|date',
            'truck_no'                     => 'required|string|max:50',
            'bill_no'                      => 'required|string|max:50',
            'description'                  => 'nullable|string',
            'details'                      => 'required|array|min:1',
            'details.*.control_account_id' => 'required|integer',
            'details.*.qty'                => 'required|numeric|min:0',
            'details.*.unit'               => 'required|min:1',
            'details.*.price'              => 'required|numeric|min:0',
            'details.*.narration'          => 'nullable|string',
        ]);

        DB::transaction(function () use ($validated, $voucher) {

            $totalAmount = collect($validated['details'])->sum(function ($item) {
                return $item['qty'] * $item['price'];
            });

            $voucher->update([
                'voucher_date' => $validated['voucher_date'],
                'truck_no'     => $validated['truck_no'],
                'bill_no'      => $validated['bill_no'],
                'supplier_id'  => $validated['supplier_id'],
                'total_amount' => $totalAmount,
                'description'  => $validated['description'] ?? null,
            ]);

            // Delete old details and re-insert fresh
            $voucher->details()->delete();

            foreach ($validated['details'] as $row) {
                $voucher->details()->create([
                    'control_account_id' => $row['control_account_id'],
                    'qty'                => $row['qty'],
                    'unit'               => $row['unit'],
                    'price'              => $row['price'],
                    'amount'             => $row['qty'] * $row['price'],
                    'narration'          => $row['narration'] ?? null,
                ]);
            }
        });

        return redirect(route('service_vouchers.index'))
            ->with('alert', 'Service Voucher Updated Successfully')
            ->with('alert-class', 'success');
    }

    public function destroy($id)
    {
        $id = decrypt($id);
        $serviceVou = ServiceVoucher::find($id);
        $serviceVou->delete();
        return redirect()->route('service_vouchers.index')->with('success', 'Voucher Deleted');
    }
}
